from collections import OrderedDict

from codeable_detectors.utils import update_keyword_args


class Evidence(object):
    def __init__(self, matches):
        if not isinstance(matches, list):
            matches = [matches]
        self.matches = matches
        self._hasFailed = False

    def __str__(self):
        return (str(self.__class__) + ": Matches = [" +
                ", ".join(str(m) for m in self.matches) + "]")

    def has_failed(self):
        return self._hasFailed

    def no_matches_found(self):
        return not self.matches

    def has_succeeded(self):
        return not self._hasFailed and self.matches


class FailedEvidence(Evidence):
    def __init__(self, message):
        super().__init__([])
        self.msg = message
        self._hasFailed = True

    def __str__(self):
        return "Failed Evidence = " + self.msg

    def prepend(self, prepend_message):
        self.msg = prepend_message + self.msg


def _get_evidences_list(evidences):
    if isinstance(evidences, Evidences):
        evidences_list = evidences.get()
    elif isinstance(evidences, list):
        evidences_list = evidences
    else:
        evidences_list = [evidences]
    return evidences_list


class Evidences(object):
    def __init__(self, evidences=None):
        if evidences is None:
            self.evidences = []
        else:
            self.evidences = _get_evidences_list(evidences)

    def add(self, evidences):
        self.evidences.extend(_get_evidences_list(evidences))

    def add_only_failed(self, evidences, prepend_to_failed_string=None):
        evidences_list = _get_evidences_list(evidences)
        failed_evidences = [e for e in evidences_list if isinstance(e, FailedEvidence)]
        if prepend_to_failed_string:
            for failed in failed_evidences:
                failed.prepend(prepend_to_failed_string)
        self.evidences.extend(failed_evidences)

    def get(self):
        return self.evidences

    def get_all_matches(self):
        return [m for em in [e.matches for e in self.evidences] for m in em]

    def have_failed(self):
        for e in self.evidences:
            if isinstance(e, FailedEvidence):
                return True
        return False

    def is_empty(self):
        if self.evidences:
            return False
        return True


class NamedEvidence(Evidence):
    def __init__(self, name, matches):
        super().__init__(matches)
        self.name = name


class ComponentEvidence(NamedEvidence):
    def __init__(self, matches, **kwargs):
        options = update_keyword_args({
            'name': None,
            'technology_types': [],
            'component_types': [],
            # possible link types can be collected on the component if its matches might
            # be considered as enough evidence for a link as well (e.g., for a web server
            # offering the component already implies possible links to the web clients)
            'link_types': []}, kwargs)
        super().__init__(options["name"], matches)
        self.component_types = options["component_types"]
        self.technology_types = options["technology_types"]
        self.link_types = options["link_types"]

    def set_properties(self, detector_name=None,
                       detector_component_types=None, detector_link_types=None,
                       detector_technology_types=None, kwargs=None):
        if detector_component_types is None:
            detector_component_types = []
        if detector_link_types is None:
            detector_link_types = []
        if detector_technology_types is None:
            detector_technology_types = []
        if kwargs is None:
            kwargs = {}

        if not isinstance(detector_component_types, list):
            detector_component_types = [detector_component_types]
        if not isinstance(detector_link_types, list):
            detector_link_types = [detector_link_types]
        if not isinstance(detector_technology_types, list):
            detector_technology_types = [detector_technology_types]

        options = update_keyword_args({'name': None, 'component_types': [], 'link_types': [],
                                     'technology_types': []}, kwargs)

        name = None
        if detector_name:
            name = detector_name
        if options["name"]:
            name = options["name"]
        self.name = name

        new_component_types = detector_component_types + options["component_types"]
        for new_component_type in new_component_types:
            if new_component_type not in self.component_types:
                self.component_types.append(new_component_type)

        new_link_types = detector_link_types + options["link_types"]
        for new_link_type in new_link_types:
            if new_link_type not in self.link_types:
                self.link_types.append(new_link_type)

        new_technology_types = detector_technology_types + options["technology_types"]
        for new_technology_type in new_technology_types:
            if new_technology_type not in self.technology_types:
                self.technology_types.append(new_technology_type)
        return self


class ServiceEvidence(ComponentEvidence):
    def __init__(self, matches, **kwargs):
        super().__init__(matches, **kwargs)
        self.component_types = ["service"] + self.component_types


class LinkEvidence(Evidence):
    def __init__(self, matches, **kwargs):
        options = update_keyword_args({'technology_types': [],
                                     'link_types': []}, kwargs)
        super().__init__(matches)
        self.link_types = options["link_types"]
        self.technology_types = options["technology_types"]
        self.tagged_values = {}

    def set_properties(self, detector_link_types=None,
                       detector_technology_types=None, detector_tagged_values=None, kwargs=None):
        if detector_link_types is None:
            detector_link_types = []
        if detector_technology_types is None:
            detector_technology_types = []
        if detector_tagged_values is None:
            detector_tagged_values = {}
        if kwargs is None:
            kwargs = {}

        if not isinstance(detector_link_types, list):
            detector_link_types = [detector_link_types]
        if not isinstance(detector_technology_types, list):
            detector_technology_types = [detector_technology_types]

        options = update_keyword_args({'component_types': [], 'link_types': [],
                                     'technology_types': []}, kwargs)

        new_link_types = detector_link_types + options["link_types"]
        for new_link_type in new_link_types:
            if new_link_type not in self.link_types:
                self.link_types.append(new_link_type)

        new_technology_types = detector_technology_types + options["technology_types"]
        for new_technology_type in new_technology_types:
            if new_technology_type not in self.technology_types:
                self.technology_types.append(new_technology_type)

        self.tagged_values = detector_tagged_values

        return self

    def update(self, link_evidence):
        self.matches = list(OrderedDict.fromkeys(self.matches + link_evidence.matches))
        self.link_types = list(OrderedDict.fromkeys(self.link_types + link_evidence.link_types))
        self.technology_types = list(OrderedDict.fromkeys(self.technology_types + link_evidence.technology_types))
        self.tagged_values.update(link_evidence.taggedValues)
